<?php

if (!defined('ABSPATH')) {
    exit;
}

class Movapay_Client {
    public static function get_option($key, $default = '') {
        $opts = get_option('movapay_gateway_options', []);
        return isset($opts[$key]) && $opts[$key] !== '' ? $opts[$key] : $default;
    }

    public static function base_url() {
        $base = trim(self::get_option('base_url', 'https://movapay.net'));
        return rtrim($base, '/');
    }

    public static function token_endpoint() {
        return self::base_url() . '/o/token/';
    }

    public static function payment_link_endpoint($integration_id) {
        $integration_id = rawurlencode($integration_id);
        return self::base_url() . '/api/v1/developer/api/integrations/' . $integration_id . '/payment-links/';
    }

    /**
     * Returns array('access_token' => string) or WP_Error
     */
    public static function get_access_token() {
        $cached = get_transient('movapay_access_token');
        if (is_array($cached) && !empty($cached['access_token'])) {
            return $cached;
        }

        $client_id = self::get_option('client_id');
        $client_secret = self::get_option('client_secret');
        if (!$client_id || !$client_secret) {
            return new WP_Error('movapay_missing_credentials', 'Movapay: client_id/client_secret manquants.');
        }

        $body = [
            'grant_type' => 'client_credentials',
            'client_id' => $client_id,
            'client_secret' => $client_secret,
        ];

        $resp = wp_remote_post(self::token_endpoint(), [
            'timeout' => 20,
            'headers' => [
                'Accept' => 'application/json',
            ],
            'body' => $body,
        ]);

        if (is_wp_error($resp)) {
            return $resp;
        }

        $code = wp_remote_retrieve_response_code($resp);
        $json = json_decode(wp_remote_retrieve_body($resp), true);

        if ($code < 200 || $code >= 300) {
            $msg = is_array($json) && isset($json['error_description']) ? $json['error_description'] : 'Erreur token OAuth2.';
            return new WP_Error('movapay_token_error', 'Movapay: ' . $msg, ['http_code' => $code, 'response' => $json]);
        }

        if (!is_array($json) || empty($json['access_token'])) {
            return new WP_Error('movapay_token_invalid', 'Movapay: réponse token invalide.', ['response' => $json]);
        }

        $expires_in = isset($json['expires_in']) ? intval($json['expires_in']) : 3600;
        // Cache with small safety margin
        set_transient('movapay_access_token', ['access_token' => $json['access_token']], max(60, $expires_in - 60));

        return ['access_token' => $json['access_token']];
    }

    /**
     * Create payment link.
     * Returns array('payment_url' => string, 'token' => string, ...) or WP_Error
     */
    public static function create_payment_link($payload) {
        $integration_id = self::get_option('integration_id');
        if (!$integration_id) {
            return new WP_Error('movapay_missing_integration', 'Movapay: integration_id manquant.');
        }

        $token = self::get_access_token();
        if (is_wp_error($token)) {
            return $token;
        }

        $resp = wp_remote_post(self::payment_link_endpoint($integration_id), [
            'timeout' => 25,
            'headers' => [
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $token['access_token'],
            ],
            'body' => wp_json_encode($payload),
        ]);

        if (is_wp_error($resp)) {
            return $resp;
        }

        $code = wp_remote_retrieve_response_code($resp);
        $json = json_decode(wp_remote_retrieve_body($resp), true);

        if ($code < 200 || $code >= 300) {
            $msg = is_array($json) && isset($json['error']) ? $json['error'] : 'Erreur API Movapay.';
            return new WP_Error('movapay_api_error', 'Movapay: ' . $msg, ['http_code' => $code, 'response' => $json]);
        }

        if (!is_array($json) || empty($json['success'])) {
            return new WP_Error('movapay_api_invalid', 'Movapay: réponse API invalide.', ['response' => $json]);
        }

        return $json;
    }
}


