import time
import requests

from odoo import api, fields, models, _
from odoo.exceptions import UserError


class AccountMove(models.Model):
    _inherit = "account.move"

    movapay_payment_url = fields.Char(string="Movapay Payment URL", copy=False, readonly=True)
    movapay_payment_token = fields.Char(string="Movapay Token", copy=False, readonly=True)
    movapay_payment_status = fields.Selection(
        [
            ("initiating", "Initiating"),
            ("created", "Created"),
            ("pending", "Pending"),
            ("success", "Success"),
            ("failed", "Failed"),
        ],
        string="Movapay Status",
        copy=False,
        readonly=True,
    )

    def _movapay_cfg(self):
        icp = self.env["ir.config_parameter"].sudo()
        base_url = (icp.get_param("movapay.base_url") or "https://movapay.net").rstrip("/")
        return {
            "base_url": base_url,
            "client_id": icp.get_param("movapay.client_id") or "",
            "client_secret": icp.get_param("movapay.client_secret") or "",
            "master_key": icp.get_param("movapay.master_key") or "",
            "integration_id": icp.get_param("movapay.integration_id") or "",
            "default_return_url": icp.get_param("movapay.default_return_url") or "",
            "default_cancel_url": icp.get_param("movapay.default_cancel_url") or "",
        }

    def _movapay_token(self):
        icp = self.env["ir.config_parameter"].sudo()
        cached = icp.get_param("movapay.access_token")
        cached_exp = icp.get_param("movapay.access_token_exp")
        if cached and cached_exp and int(cached_exp) > int(time.time()) + 30:
            return cached

        cfg = self._movapay_cfg()
        if not cfg["client_id"] or not cfg["client_secret"]:
            raise UserError(_("Movapay: Client ID/Secret manquants (Paramètres)."))

        url = f'{cfg["base_url"]}/o/token/'
        data = {
            "grant_type": "client_credentials",
            "client_id": cfg["client_id"],
            "client_secret": cfg["client_secret"],
        }
        r = requests.post(url, data=data, timeout=20)
        if r.status_code < 200 or r.status_code >= 300:
            raise UserError(_("Movapay: erreur token OAuth2 (%s)") % r.text)

        js = r.json()
        token = js.get("access_token")
        if not token:
            raise UserError(_("Movapay: réponse token invalide."))

        expires_in = int(js.get("expires_in") or 3600)
        icp.set_param("movapay.access_token", token)
        icp.set_param("movapay.access_token_exp", str(int(time.time()) + expires_in))
        return token

    def action_movapay_generate_link(self):
        for move in self:
            if move.state != "posted":
                raise UserError(_("Veuillez d'abord valider (poster) la facture."))
            if move.move_type not in ("out_invoice", "out_refund"):
                raise UserError(_("Movapay: uniquement pour les factures client."))

            cfg = move._movapay_cfg()
            if not cfg["integration_id"]:
                raise UserError(_("Movapay: Integration ID manquant (Paramètres)."))

            token = move._movapay_token()
            url = f'{cfg["base_url"]}/api/v1/developer/api/integrations/{cfg["integration_id"]}/payment-links/'

            payload = {
                "amount": float(move.amount_total),
                "currency": move.currency_id.name,
                "description": f"Facture {move.name}",
                "customer_email": move.partner_id.email or "",
                "customer_phone": move.partner_id.phone or "",
            }

            if cfg["default_return_url"]:
                payload["return_url"] = cfg["default_return_url"]
            if cfg["default_cancel_url"]:
                payload["cancel_url"] = cfg["default_cancel_url"]

            headers = {
                "Authorization": f"Bearer {token}",
                "Accept": "application/json",
                "Content-Type": "application/json",
            }
            r = requests.post(url, json=payload, headers=headers, timeout=25)
            if r.status_code < 200 or r.status_code >= 300:
                raise UserError(_("Movapay: erreur création lien (%s)") % r.text)

            js = r.json()
            if not js.get("success"):
                raise UserError(_("Movapay: réponse invalide (%s)") % r.text)

            payment_url = js.get("payment_url") or js.get("payment_link")
            if not payment_url:
                raise UserError(_("Movapay: payment_url manquant."))

            move.movapay_payment_url = payment_url
            move.movapay_payment_token = js.get("token") or ""
            move.movapay_payment_status = js.get("status") or "created"
        return True


