import hmac
import hashlib
import json

from odoo import http
from odoo.http import request


class MovapayWebhookController(http.Controller):
    @http.route("/movapay/webhook", type="http", auth="public", methods=["POST"], csrf=False)
    def movapay_webhook(self, **kwargs):
        raw = request.httprequest.data or b""
        sig = request.httprequest.headers.get("X-Signature")

        icp = request.env["ir.config_parameter"].sudo()
        master_key = icp.get_param("movapay.master_key") or ""

        if not master_key or not sig:
            return request.make_response("missing_signature_or_key", headers=[("Content-Type", "text/plain")], status=400)

        computed = hmac.new(master_key.encode("utf-8"), raw, hashlib.sha512).hexdigest()
        if not hmac.compare_digest(computed, sig):
            return request.make_response("invalid_signature", headers=[("Content-Type", "text/plain")], status=403)

        try:
            payload = json.loads(raw.decode("utf-8"))
        except Exception:
            return request.make_response("invalid_json", headers=[("Content-Type", "text/plain")], status=400)

        token = ((payload.get("custom_data") or {}).get("token")) or ""
        status_val = payload.get("status") or ""
        tx_ref = payload.get("transaction_ref") or ""

        if token:
            moves = request.env["account.move"].sudo().search([("movapay_payment_token", "=", token)], limit=5)
            for move in moves:
                try:
                    move.write({"movapay_payment_status": status_val or move.movapay_payment_status})
                    if hasattr(move, "message_post"):
                        move.message_post(
                            body=f"Webhook Movapay reçu: status={status_val} transaction_ref={tx_ref}",
                            message_type="comment",
                        )
                except Exception:
                    # Never fail webhook response for a local posting issue
                    pass

        return request.make_response("ok", headers=[("Content-Type", "text/plain")], status=200)


